//--------------------------------------------
// Christopher Ravenscroft 2009
// Game Demo
//--------------------------------------------

/*Note: World Matrix is not used in calculations as it is always identity [opt]*/
/*Note: WVIP not used on normals due to lack of non-uniform scaling etc...[opt]*/

//--------------------------------------------
// Global variables
//--------------------------------------------
uniform extern float4x4 g_worldMatrix;
uniform extern float4x4 g_worldViewProjectionMatrix;
uniform extern matrix	g_matTexture;

uniform extern float3 g_cameraPos;

//Texture variables
uniform extern texture tShadowMap;

uniform extern texture tColourTex;
uniform extern texture tNormalMap;

uniform extern texture tSplatTex;
uniform extern texture tFloorTex1;
uniform extern texture tFloorTex2;
uniform extern texture tFloorTex3;

uniform extern textureCUBE tCubeTex;

// Lighting variables
uniform extern float4 g_lightAmbient;

uniform extern float4 g_materialAmbient;
uniform extern float4 g_materialDiffuse;
uniform extern float4 g_materialSpecular;
uniform extern float g_materialShininess;

const float3 lightDir = -float3(-0.75f, -0.97f, 0.75f);

// Shadow mapping variables
uniform extern bool g_bPCF;	// Toggles PCF or hard shadows
uniform extern float bias;		// shadow map bias: changes on shadow map size to remove artifacts
uniform extern float g_fTexelSize;	// Used for PCF


//--------------------------------------------
// Texture Samplers
//--------------------------------------------

samplerCUBE CubeSampler = 
sampler_state
{
	texture = <tCubeTex>;

	MipFilter = Linear;
	MinFilter = Linear;
	MagFilter = Linear;
	MaxAnisotropy = 1;

	AddressU = Clamp;
	AddressV = Clamp;
};

sampler ShadowMap = 
sampler_state
{
	texture = <tShadowMap>;

	MipFilter = None;
   	MinFilter = None;
    	MagFilter = None;
    	//MaxAnisotropy = 0;
	
	AddressU = Clamp;
	AddressV = Clamp;
};

sampler ColourTex = 
sampler_state
{
	texture = <tColourTex>;

    	MinFilter = Anisotropic;
   	MagFilter = Linear;
   	MipFilter = Linear;
    	MaxAnisotropy = 16;
	
	AddressU = Wrap;//clamp
	AddressV = Wrap;
};

sampler NormalMap = 
sampler_state
{
	texture = <tNormalMap>;

   	MinFilter = Anisotropic;
    	MagFilter = Linear;
    	MipFilter = Linear;
   	MaxAnisotropy = 16;
	
	AddressU = Wrap;//clamp
	AddressV = Wrap;
};

sampler SplatTex = 
sampler_state
{
	texture = <tSplatTex>;

	MipFilter = Linear;
    	MinFilter = Anisotropic;
    	MagFilter = Linear;
   	MaxAnisotropy = 2;
	
	AddressU = Clamp;
	AddressV = Clamp;
};

sampler FloorTex1 = 
sampler_state
{
	texture = <tFloorTex1>;

	MipFilter = Linear;
    	MinFilter = Anisotropic;
    	MagFilter = Linear;
   	MaxAnisotropy = 4;
	
	AddressU = Wrap;
	AddressV = Wrap;
};

sampler FloorTex2 = 
sampler_state
{
	texture = <tFloorTex2>;

	MipFilter = Linear;
    	MinFilter = Anisotropic;
    	MagFilter = Linear;
   	MaxAnisotropy = 4;
	
	AddressU = Wrap;
	AddressV = Wrap;
};

sampler FloorTex3 = 
sampler_state
{
	texture = <tFloorTex3>;

	MipFilter = Linear;
    	MinFilter = None;
    	MagFilter = Linear;
   	MaxAnisotropy = 4;
	
	AddressU = Wrap;
	AddressV = Wrap;
};


//--------------------------------------------
// Vertex shaders
//--------------------------------------------
struct VS_INPUT
{
	float3 position 	: POSITION0;	// stream 1
	float3 normal 		: NORMAL;		// stream 1
	float3 tangent		: TANGENT;		// stream 1
	float2 texCoord 	: TEXCOORD0;	// stream 1
	float4 instancePos	: POSITION1;	// stream 2: packed as (x, y, z, rotation)
};
	
struct VS_OUTPUT_DIR
{
	float4 vPosition 	: POSITION0;
	float2 vTexCoord 	: TEXCOORD0;
	float3 viewDir	 	: TEXCOORD1;
	float3 lightDir 	: TEXCOORD2;
	float4 vProjCoord 	: TEXCOORD3;
	float4 vScreenCoord : TEXCOORD4;
};

VS_OUTPUT_DIR VS_DirLighting(VS_INPUT IN)
{
	VS_OUTPUT_DIR OUT;

	// Create the rotation matrix
	float3x3 rMatrix = float3x3(cos(IN.instancePos.w), 0.0f, -sin(IN.instancePos.w),
					0.0f, 1.0f, 0.0f,
					sin(IN.instancePos.w), 0.0f, cos(IN.instancePos.w));	
					
	// Rotate the model
	float3 vRotatedPos = mul(IN.position, rMatrix);

	// Translate the model
	vRotatedPos += IN.instancePos.xyz;

	// Transform the position
	OUT.vPosition = mul(float4(vRotatedPos, 1.0f), g_worldViewProjectionMatrix);

	// Calculate lighting (specular)
	float3 viewDir = g_cameraPos - vRotatedPos;

	// Multiply normal,tangent by the rotated world matrix
	float3 n = mul(IN.normal, rMatrix);
	float3 t = mul(IN.tangent, rMatrix);
	float3 b = cross(n, t);	

	float3x3 tbnMatrix = float3x3(t.x, b.x, n.x,
	                              t.y, b.y, n.y,
	                              t.z, b.z, n.z); 

	
	// Output the projective texture coordinates for the shadow map
	OUT.vProjCoord = mul( float4(vRotatedPos, 1.0f), g_matTexture );

	// Output the screen-space texture coordinates for the shadow map
	OUT.vScreenCoord.x = ( OUT.vPosition.x * 0.5 + OUT.vPosition.w * 0.5 );
	OUT.vScreenCoord.y = ( OUT.vPosition.w * 0.5 - OUT.vPosition.y * 0.5 );
	OUT.vScreenCoord.z = OUT.vPosition.w;
	OUT.vScreenCoord.w = OUT.vPosition.w;

	OUT.vTexCoord = IN.texCoord;	// Pass through
	OUT.viewDir = mul(viewDir, tbnMatrix);
	OUT.lightDir = mul(lightDir, tbnMatrix);

	return OUT;
}


struct VS_INPUT_FLOOR
{
	float3 position : POSITION0;
	float3 normal : NORMAL;
	float2 texCoord : TEXCOORD0;
	float2 texCoordSplat : TEXCOORD1;
};

struct VS_OUTPUT_FLOOR
{
	float4 vPosition		: POSITION0;
	float2 vTexCoord		: TEXCOORD0;
	float2 vTexCoordSplat	: TEXCOORD1;
	float4 vProjCoord 		: TEXCOORD2;
	float4 vScreenCoord 	: TEXCOORD3;
	float3 vNormal			: TEXCOORD4;
	float3 vLightVec		: TEXCOORD5;
};

// Scene vertex shader
VS_OUTPUT_FLOOR VS_Floor( VS_INPUT_FLOOR IN )
{
	VS_OUTPUT_FLOOR OUT;

	// Output the transformed position
	OUT.vPosition = mul( float4(IN.position,1.0f), g_worldViewProjectionMatrix );

	// Output the texture coordinates
	OUT.vTexCoord = IN.texCoord;
	OUT.vTexCoordSplat = IN.texCoordSplat;

	// Output the projective texture coordinates for the shadow map
	OUT.vProjCoord = mul( float4(IN.position, 1.0f), g_matTexture );

	// Output the screen-space texture coordinates for the shadow map
	OUT.vScreenCoord.x = ( OUT.vPosition.x * 0.5 + OUT.vPosition.w * 0.5 );
	OUT.vScreenCoord.y = ( OUT.vPosition.w * 0.5 - OUT.vPosition.y * 0.5 );
	OUT.vScreenCoord.z = OUT.vPosition.w;
	OUT.vScreenCoord.w = OUT.vPosition.w;

	// Already in world pos
	float3 vWorldPos = IN.position;

	// Normal is already world space normal
	OUT.vNormal = IN.normal;

	// light vector
	OUT.vLightVec = -float3(-1.0f, -0.95f, 0.95f);
	
	return OUT;
}

struct VS_INPUT_WALLS
{
	float3 position : POSITION0;
	float3 normal : NORMAL;
	float3 tangent : TANGENT;
	float2 texCoord : TEXCOORD0;
};
struct VS_OUTPUT_WALLS
{
	float4 vPosition		: POSITION0;
	float2 texCoord			: TEXCOORD0;
	float4 vProjCoord 		: TEXCOORD1;
	float4 vScreenCoord 		: TEXCOORD2;
	float3 viewDir			: TEXCOORD3;
	float3 lightDir			: TEXCOORD4;
};

// Scene vertex shader
VS_OUTPUT_WALLS VS_Walls( VS_INPUT_WALLS IN )
{
	VS_OUTPUT_WALLS OUT;

	// Output the transformed position
	OUT.vPosition = mul( float4(IN.position,1.0f), g_worldViewProjectionMatrix );
	
	OUT.texCoord = IN.texCoord;	// Pass through

	// Output the projective texture coordinates for the shadow map
	OUT.vProjCoord = mul( float4(IN.position, 1.0f), g_matTexture );

	// Output the screen-space texture coordinates for the shadow map
	OUT.vScreenCoord.x = ( OUT.vPosition.x * 0.5 + OUT.vPosition.w * 0.5 );
	OUT.vScreenCoord.y = ( OUT.vPosition.w * 0.5 - OUT.vPosition.y * 0.5 );
	OUT.vScreenCoord.z = OUT.vPosition.w;
	OUT.vScreenCoord.w = OUT.vPosition.w;

	// Pos is already in world space
	float3 vWorldPos = IN.position;

	// Calculate lighting (specular)
	float3 viewDir = g_cameraPos - vWorldPos;

	// Create TBN Matrix
	float3 n = IN.normal;
	float3 t = IN.tangent;

	float3 b = cross(n, t);
	float3x3 tbnMatrix = float3x3(t.x, b.x, n.x,
	                              t.y, b.y, n.y,
	                              t.z, b.z, n.z);

	OUT.viewDir = mul(viewDir, tbnMatrix);
	OUT.lightDir = mul(lightDir, tbnMatrix);
	
	return OUT;
}

// SHADOW VERTEX SHADERS
struct VS_INPUT_SHADOW
{
	float3 vPosition : POSITION;
};

struct VS_OUTPUT_SHADOW
{
	float4 vPosition	: POSITION;
	float  fDepth		: TEXCOORD0;
};

VS_OUTPUT_SHADOW VS_Shadow( VS_INPUT_SHADOW IN)
{
	VS_OUTPUT_SHADOW OUT;

	// Output the transformed position
	OUT.vPosition = mul( float4(IN.vPosition, 1.0f), g_worldViewProjectionMatrix );
	// Output the scene depth
	OUT.fDepth = OUT.vPosition.z;

	return OUT;
}

struct VS_INPUT_SHADOW_Inst
{
	float3 vPosition : POSITION;	// Stream 1
	float4 instancePos : POSITION1; // Stream 2
};


VS_OUTPUT_SHADOW VS_ShadowInst( VS_INPUT_SHADOW_Inst IN)
{
	VS_OUTPUT_SHADOW OUT;

	// Create the rotation matrix
	float3x3 rMatrix = float3x3(cos(IN.instancePos.w), 0.0f, -sin(IN.instancePos.w),
					0.0f, 1.0f, 0.0f,
					sin(IN.instancePos.w), 0.0f, cos(IN.instancePos.w));	

	// Rotate the model
	float3 vRotatedPos = mul(IN.vPosition, rMatrix);

	// Translate the model
	vRotatedPos += IN.instancePos.xyz;

	// Transform the position
	OUT.vPosition = mul(float4(vRotatedPos, 1.0f), g_worldViewProjectionMatrix);

	// Output the scene depth
	OUT.fDepth = OUT.vPosition.z;

	return OUT;
}


struct VS_INPUT_SKYBOX
{
	float4 vPosition : POSITION;
	float3 vNormal : NORMAL;
};

struct VS_OUTPUT_SKYBOX
{
	float4 vPosition : POSITION;
	float3 vViewDirection : TEXCOORD0;
};

VS_OUTPUT_SKYBOX VS_Skybox( VS_INPUT_SKYBOX IN)
{
	VS_OUTPUT_SKYBOX OUT;

	// Output the transformed position
	OUT.vPosition = mul( IN.vPosition, g_worldViewProjectionMatrix );
	
	// Calculate the view direction
	OUT.vViewDirection = g_cameraPos - mul(IN.vPosition, g_worldMatrix).xyz;

	return OUT;
}

//-----------------------------------------------------------------------------
// Helper functions.
//-----------------------------------------------------------------------------

// Shadow fragment function
float GetShadowTerm(float4 vProjCoord)
{
	// Shadow the object
	float fShadowTerm = 0.0f;

	if( g_bPCF )
	{
		// Variable to hold texture offsets for 3x3 PCF kernel
		float4 vTexCoords[9];

		// Create texel offsets
		vTexCoords[0] = vProjCoord;
		vTexCoords[1] = vProjCoord + float4( -g_fTexelSize, 0.0f, 0.0f, 0.0f );
		vTexCoords[2] = vProjCoord + float4(  g_fTexelSize, 0.0f, 0.0f, 0.0f );
		vTexCoords[3] = vProjCoord + float4( 0.0f, -g_fTexelSize, 0.0f, 0.0f );
		vTexCoords[6] = vProjCoord + float4( 0.0f,  g_fTexelSize, 0.0f, 0.0f );
		vTexCoords[4] = vProjCoord + float4( -g_fTexelSize, -g_fTexelSize, 0.0f, 0.0f );
		vTexCoords[5] = vProjCoord + float4(  g_fTexelSize, -g_fTexelSize, 0.0f, 0.0f );
		vTexCoords[7] = vProjCoord + float4( -g_fTexelSize,  g_fTexelSize, 0.0f, 0.0f );
		vTexCoords[8] = vProjCoord + float4(  g_fTexelSize,  g_fTexelSize, 0.0f, 0.0f );

		// Depth test the sample region
		for( int i = 0; i < 9; ++i )
		{
			float A = tex2Dproj( ShadowMap, vTexCoords[i] ).r;
			float B = vProjCoord.z - bias;
		
			// Perform depth test
			fShadowTerm += A < B ? 0.3f : 1.0f;
		}
	
		// Get the average
		fShadowTerm = fShadowTerm / 9.0f;
	}
	else
	{
		float A = tex2Dproj( ShadowMap, vProjCoord ).r;
		float B = vProjCoord.z - 0.0001f;
		
		// Perform depth test
		fShadowTerm += A < B ? 0.3f : 1.0f;
	}

	return fShadowTerm;
}

//-----------------------------------------------------------------------------
// Pixel Shaders.
//-----------------------------------------------------------------------------

// Instanced objects:
float4 PS_DirLighting(VS_OUTPUT_DIR IN) : COLOR
{
	float3 n = normalize(2.0f * tex2D(NormalMap, IN.vTexCoord).rgb - 1.0f);
   	float3 v = normalize(IN.viewDir);
   	float3 l = normalize(IN.lightDir);
    
	// Faces facing the light
    float nDotL = saturate(dot(n, l));

	// Grab the shadow term
	float fShadowTerm = GetShadowTerm(IN.vProjCoord);

	// Calculate the lighting term
	float3 r = -reflect(l, n);  // Reflect
	float4 spec_pow = (nDotL == 0.0f) ? 0.0f : pow(saturate(dot(r, v)), g_materialShininess); // R.V^n
	float4 color = (g_lightAmbient * g_materialAmbient) + (((g_materialDiffuse * nDotL) + (g_materialSpecular * spec_pow)) * fShadowTerm);

	// Multiply lighting by the texture colour
	color *= tex2D(ColourTex, IN.vTexCoord);

	return color;
}



// Floor pixel shader
float4 PS_Floor( VS_OUTPUT_FLOOR IN ) : COLOR0
{
	// Normalize the normal, light and eye vectors
	IN.vNormal	 = normalize( IN.vNormal );
	IN.vLightVec = normalize( IN.vLightVec );

	float3 mask = tex2D(SplatTex, IN.vTexCoordSplat).rgb;
	float3 texColour = tex2D(FloorTex1, IN.vTexCoord).rgb;
	float3 texColour2= tex2D(FloorTex2, IN.vTexCoord).rgb;
	float3 texColour3 = tex2D(FloorTex3, IN.vTexCoord).rgb;

	//combine textures for texture splatting
	texColour = (mask.r * texColour2) + ((1 - mask.r) * texColour);
	texColour = (mask.g * texColour3) + ((1 - mask.g) * texColour);

	// Ambient light
	float3 ambient = (float3(0.35,0.35,0.35));
	
	// Diffuse light
	float3 diffuse = (float3(0.4,0.4,0.4));

	// Compute the diffuse and specular lighting terms
	float NdotL  = max( dot( IN.vNormal, IN.vLightVec ), 0 );

	// Grab the shadow term
	float fShadowTerm = GetShadowTerm(IN.vProjCoord);

	// Final Colour
	texColour *= (g_lightAmbient * ambient) + (diffuse * NdotL * fShadowTerm );
	
	return float4(texColour, 1.0);
}


// Floor walls pixel shader
float4 PS_Walls( VS_OUTPUT_WALLS IN ) : COLOR0
{
	float3 n = normalize(2.0f * tex2D(NormalMap, IN.texCoord).rgb - 1.0f);
	float3 v = normalize(IN.viewDir);
   	float3 l = normalize(IN.lightDir);

	float nDotL = saturate(dot(n, l));	// Diffuse

	// Grab the shadow term
	float fShadowTerm = GetShadowTerm(IN.vProjCoord);

	// Specular
	float3 r = -reflect(l, n);  // Reflect
	float4 spec_pow = pow(saturate(dot(r, v)), g_materialShininess); // R.V^n

	// Calculate the lighting term
	float4 Colour = (g_lightAmbient * g_materialAmbient ) + (((g_materialDiffuse * nDotL) + (g_materialSpecular * spec_pow)) * fShadowTerm);


	// Multiply lighting by texture colour
	Colour *= tex2D(ColourTex, IN.texCoord);
	return Colour;
}

// SHADOW PIXEL SHADER
float4  PS_Shadow( VS_OUTPUT_SHADOW IN ) : COLOR0
{
	// Output the scene depth
	//return float4( IN.fDepth, IN.fDepth, IN.fDepth, 1.0f );
	return IN.fDepth;
}


// Skybox PS
float4 PS_Skybox(VS_OUTPUT_SKYBOX IN) : COLOR
{
	float3 vViewDirection = -normalize(IN.vViewDirection);
	float4 Colour = texCUBE(CubeSampler, vViewDirection); 
	return Colour;
}

// Z Pass PIXEL SHADER
float4  PS_ZPass( /*VS_OUTPUT_SHADOW IN*/ ) : COLOR0
{
	return float4(0.0f, 0.0f, 0.0f, 1.0f );
}

//--------------------------------------------
// Techniques
//--------------------------------------------
technique InstancedObjects
{
	pass P0
	{
		Lighting	= False;
		CullMode	= CCW;
		ZENABLE 	= True;
		ZWriteEnable    = False;
		ColorWriteEnable    = RED|GREEN|BLUE|ALPHA;
		ZFunc 		= LessEqual;

		VertexShader = compile vs_3_0 VS_DirLighting();
		PixelShader  = compile ps_3_0 PS_DirLighting();
	}
}

technique Floor
{
	pass P0
	{
		Lighting	= False;
		CullMode	= CCW;
		ZENABLE		= True;
		ZFunc 		= LessEqual;	
		//ZWriteEnable    = False;
		//ColorWriteEnable    = RED|GREEN|BLUE|ALPHA;	

		VertexShader = compile vs_3_0 VS_Floor();
		PixelShader  = compile ps_3_0 PS_Floor();
	}
}

technique Walls
{
	pass P0
	{
		Lighting	= False;
		CullMode	= CCW;
		ZENABLE		= True;	
		ZFunc 		= LessEqual;	
		//ZWriteEnable    = False;
		//ColorWriteEnable    = RED|GREEN|BLUE|ALPHA;

		VertexShader = compile vs_3_0 VS_Walls();
		PixelShader  = compile ps_3_0 PS_Walls();
	}
}

technique Shadow
{
	pass P0
	{
		Lighting 	= False;
		CullMode	= None;
		
		VertexShader = compile vs_3_0 VS_Shadow();
		PixelShader  = compile ps_3_0 PS_Shadow();
	}
}
technique ShadowInst
{
	pass P0
	{
		Lighting 	= False;
		CullMode	= None;
		
		VertexShader = compile vs_3_0 VS_ShadowInst();
		PixelShader  = compile ps_3_0 PS_Shadow();
	}
}

technique Skybox
{
	pass P0
	{
		Lighting 	= False;
		CullMode	= CCW;
		ZENABLE		= False;
		//ZWriteEnable    = False;
		//ColorWriteEnable    = RED|GREEN|BLUE|ALPHA;	
		
		VertexShader = compile vs_3_0 VS_Skybox();
		PixelShader  = compile ps_3_0 PS_Skybox();
	}
}

technique InstancedZPass
{
	pass P0
	{
		Lighting 	= False;
		CullMode	= CCW;
		ZENABLE		= True;	
		ZWriteEnable    = True;
		ColorWriteEnable    = 0;
		
		VertexShader = compile vs_3_0 VS_ShadowInst();
		PixelShader = compile ps_3_0 PS_ZPass();
	}
}

technique NormalZPass
{
	pass P0
	{
		Lighting 	= False;
		CullMode	= CCW;
		ZENABLE		= True;	
		ZWriteEnable    = True;
		ColorWriteEnable    = 0;
		
		VertexShader = compile vs_3_0 VS_Shadow();
		PixelShader = compile ps_3_0 PS_ZPass();
	}
}