//--------------------------------------------------------------------------------------
// File: RenderScene.fx
//
// Chris Ravenscroft 2011
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Global variables
//--------------------------------------------------------------------------------------

#include "Material.fxh"

Texture2D g_tBufferTex0;        // r = VS distance, g = shinines, b = specular intensity, a = diffuse material index
Texture2D g_tBufferTex1;        // Normal texture for mesh
Texture2D g_tBufferTex2;        // Color texture for mesh

const float g_fRadiusBais = 0.5f;	// Reduce possible hard edges from low poly light volumes

cbuffer cbPerFrameDirLight
{
	matrix g_mInvProj;
	float3 g_vLightVec;
};

cbuffer cbPerFramePointLight
{
	matrix g_mView;
	matrix g_mProjection;
	float g_fFarClip;
};

cbuffer cbPerPointLight
{
	matrix g_mWorldPL;
	float3 g_vColourPL;
	float3 g_vPositionPLVS;
	float g_fRadiusPL;
	float g_fIntensityPL;
};

cbuffer cbPerAmbientPass
{
	float3 g_vAmbient;
};

//--------------------------------------------------------------------------------------
// Texture sampler
//--------------------------------------------------------------------------------------
SamplerState LinearSampler
{
    Filter = MIN_MAG_MIP_LINEAR;
    AddressU = Wrap;
    AddressV = Wrap;
};

SamplerState PointSampler
{
    Filter = MIN_MAG_MIP_POINT;
    AddressU = Wrap;
    AddressV = Wrap;
};

//--------------------------------------------------------------------------------------
// Vertex shader input structure
//--------------------------------------------------------------------------------------
struct VS_INPUT_DirLight
{
	float3 vPosition    : POSITION;
	float2 vTexUV       : TEXCOORD0;
};

struct VS_INPUT_PointLight
{
	float3 vPosition    : POSITION;
};

struct VS_INPUT_Ambient
{
	float3 vPosition    : POSITION;
	float2 vTexUV       : TEXCOORD0;
};

//--------------------------------------------------------------------------------------
// Pixel shader input structure
//--------------------------------------------------------------------------------------
struct PS_INPUT_DirLight
{
	float4 vPositionCS : SV_POSITION;
	float2 vTexUV      : TEXCOORD0;
	float3 vViewRay	   : TEXCOORD1;
};

struct PS_INPUT_PointLight
{
	float4 vPositionCS : SV_POSITION;
	float4 vScreenPos  : TEXCOORD0;
	float3 vPositionVS : TEXCOORD1;
};

struct PS_INPUT_Ambient
{
	float4 vPositionCS : SV_POSITION;
	float2 vTexUV      : TEXCOORD0;
};

//--------------------------------------------------------------------------------------
// Vertex Shaders
//--------------------------------------------------------------------------------------
PS_INPUT_DirLight VS_DirLight( VS_INPUT_DirLight IN )
{
	PS_INPUT_DirLight OUT;
	
	OUT.vPositionCS = float4(IN.vPosition, 1.0);
	OUT.vTexUV = IN.vTexUV;
	
	// Calculate the view space position for a full-screen quad assuming post-projection
	// Calculate the view ray
	OUT.vViewRay = mul(float4(IN.vPosition, 1.0), g_mInvProj).xyz;

	return OUT;
}

PS_INPUT_PointLight VS_PointLight( VS_INPUT_PointLight IN )
{
	PS_INPUT_PointLight OUT;
	
	// Calculate view space position for the view ray
	float4 vPositionVS = mul(float4(IN.vPosition, 1.0f), g_mWorldPL);
	vPositionVS = mul(vPositionVS, g_mView);
	OUT.vPositionVS = vPositionVS.xyz;

	// Clip space position
	float4 vPositionCS = mul(vPositionVS, g_mProjection);
	OUT.vPositionCS = vPositionCS;

	// To calculate tex coord in pixel shader
	OUT.vScreenPos = vPositionCS; 

	return OUT;
}

PS_INPUT_Ambient VS_AmbientPass( VS_INPUT_Ambient IN )
{
	PS_INPUT_Ambient OUT;
	
	OUT.vPositionCS = float4(IN.vPosition, 1.0);
	OUT.vTexUV = IN.vTexUV;

	return OUT;
}

//--------------------------------------------------------------------------------------
// Pixel Shaders
//--------------------------------------------------------------------------------------
float4 PS_DirLight( PS_INPUT_DirLight IN ) : SV_Target
{
	// Sample the textures to obtain values
	float4 rt0Tex = g_tBufferTex0.Sample( PointSampler, IN.vTexUV );
	float3 albedo = g_tBufferTex2.Sample( LinearSampler, IN.vTexUV ).rgb;
	float3 normal = g_tBufferTex1.Sample( PointSampler, IN.vTexUV ).rgb * 2.0f - 1.0f;
	
	// Calculate VS position 
	// Normalize the view ray, and apply the normalised view distance to reconstruct position
	float3 viewRay = IN.vViewRay;
	float normalizedDepth = rt0Tex.r;
	float3 positionVS = viewRay * normalizedDepth;

	// Lighting calculation
	// Diffuse
	normal = normalize(normal);
	float NdotL = dot(normal, -g_vLightVec);
	float3 diffuse = g_vDiffuse[rt0Tex.a]  * NdotL;

	float ss = (NdotL > 0) ? 1 : 0;	// Self shadow

	// Specular
	float3 v = normalize(-positionVS);
	float3 r = normalize(-reflect(-g_vLightVec, normal));  // Reflect
	float4 spec_pow = pow(saturate(dot(r, v)), rt0Tex.g); // R.V^n
	spec_pow *= rt0Tex.b;
	float3 specular = float3(0.7f, 0.7f, 0.7f) * spec_pow;

	float3 colour = (diffuse + specular) * ss;
	colour *= albedo.xyz;	

	return float4(colour, 1.0f);
}

float4 PS_PointLight( PS_INPUT_PointLight IN ) : SV_Target
{
	// Calculate texture coord
	IN.vScreenPos.xy /= IN.vScreenPos.w;
	float2 vTexUV = 0.5f * (float2(IN.vScreenPos.x, -IN.vScreenPos.y) + 1.0f);
	
	// Sample the textures to obtain values
	float4 rt0Tex = g_tBufferTex0.Sample( PointSampler, vTexUV );
	float3 albedo = g_tBufferTex2.Sample( LinearSampler, vTexUV ).rgb;
	float3 normal = g_tBufferTex1.Sample( PointSampler, vTexUV ).rgb * 2.0f - 1.0f;
	
	// Calculate VS position 
	// Extrapolate the view space position to the  far clip plane
	float3 viewRay = float3(IN.vPositionVS.xy * (g_fFarClip / IN.vPositionVS.z), g_fFarClip);

	float normalizedDepth = rt0Tex.r;
	float3 positionVS = viewRay * normalizedDepth;


	// Lighting calculation
	// Calculate the light vec, use for attenuation, then normalize the lightvec
	float3 lightVec = g_vPositionPLVS - positionVS;
	float attenuation = saturate(1.0f - length(lightVec) / (g_fRadiusPL - g_fRadiusBais));
	lightVec = normalize(lightVec);

	// Diffuse
	normal = normalize(normal);
	float NdotL = dot(normal, lightVec);
	float3 diffuse = g_vDiffuse[rt0Tex.a]  * NdotL;
	float ss = (NdotL > 0) ? 1 : 0;	// Self shadow	

	// Specular
	float3 v = normalize(-positionVS);
	float3 r = normalize(-reflect(lightVec, normal));  // Reflect
	float4 spec_pow = pow(saturate(dot(r, v)), rt0Tex.g); // R.V^n
	spec_pow *= rt0Tex.b; 
	float3 specular = float3(0.7f, 0.7f, 0.7f) * spec_pow;

	float3 colour = g_fIntensityPL * g_vColourPL * ((diffuse + specular) * ss) * attenuation;
	colour *= albedo.xyz;	

	return float4(colour, 1.0f);
}

float4 PS_AmbientPass( PS_INPUT_Ambient IN ) : SV_Target
{
	// Sample the textures to obtain values
	float3 albedo = g_tBufferTex2.Sample( LinearSampler, IN.vTexUV ).rgb;
	
	// Ambient light (utterly faked for now)
	float3 colour = g_vAmbient * albedo.xyz;

	return float4(colour, 1.0f);
}


//--------------------------------------------------------------------------------------
// Techniques
//--------------------------------------------------------------------------------------
technique10 DirLight
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_4_0, VS_DirLight() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_DirLight() ) );
    }
}

technique10 PointLight
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_4_0, VS_PointLight() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_PointLight() ) );
    }
}

technique10 AmbientPass
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_4_0, VS_AmbientPass() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_AmbientPass() ) );
    }
}