//--------------------------------------------------------------------------------------
// File: Luminance.fx
//
// Chris Ravenscroft 2011
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Global variables
//--------------------------------------------------------------------------------------


//float2 g_vGSOffsets[4];	// The offsets used by InitialGreyScaleDownSample()
float2 g_vGSOffsets[9];	// The offsets used by InitialGreyScaleDownSample()
float2 g_vDSOffsets[9];	// The offsets used by DownSample()

texture2D g_tInputTex0;		// Used in calculating average scene luminance
texture2D g_tInputTex1;		// Used in calculating adapted luminance

cbuffer cbAdaptedLuminance
{
	float g_fDT;
	float g_fTau;
};

const float3 LUMINANCE_WEIGHT = float3(0.2125f, 0.7154f, 0.0721f);
//const float3 LUMINANCE_WEIGHT = float3(0.333f, 0.333f, 0.333f);

float g_fMaxLum;


//--------------------------------------------------------------------------------------
// Texture sampler
//--------------------------------------------------------------------------------------
SamplerState LinearSampler
{
    Filter = MIN_MAG_MIP_LINEAR;
    AddressU = Clamp;
    AddressV = Clamp;
};

SamplerState PointSampler
{
    Filter = MIN_MAG_MIP_POINT;
    AddressU = Clamp;
    AddressV = Clamp;
};

//--------------------------------------------------------------------------------------
// Vertex shader input structure
//--------------------------------------------------------------------------------------
struct VS_INPUT_QUAD
{
	float3 vPosition    : POSITION;
	float2 vTexUV       : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Pixel shader input structure
//--------------------------------------------------------------------------------------
struct PS_INPUT
{
	float4 vPositionCS : SV_POSITION;
	float2 vTexUV      : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Vertex Shaders
//--------------------------------------------------------------------------------------
PS_INPUT VS_QUAD( VS_INPUT_QUAD IN )
{
	PS_INPUT OUT;
	
	OUT.vPositionCS = float4(IN.vPosition, 1.0);
	OUT.vTexUV = IN.vTexUV;

	return OUT;
}


//--------------------------------------------
// Pixel Shaders
//--------------------------------------------

float4 PS_InitialDownSample( PS_INPUT IN ) : SV_Target
{
	// Compute the average of the 4 necessary samples
	float average = 0.0f;
        float4 colour = 0.0f;
	float maxLum = g_fMaxLum;
	
        for( int i = 0; i < 9; i++ )
        {
		colour = g_tInputTex0.Sample( PointSampler, IN.vTexUV + g_vGSOffsets[i]);

		float luminance = dot( colour.rgb, LUMINANCE_WEIGHT );
		luminance = max(1e-5, luminance);

		luminance = log( 1e-5 + luminance );
		average += luminance;

		maxLum = max(maxLum, luminance);
	}

	average /= 9.0f;

	// Output the luminance to the render target
	return float4( average, maxLum, 0.0f, 1.0f );
}


float4 PS_DownSample( PS_INPUT IN ) : SV_Target
{
	// Compute the average of the 10 samples
	float2 luminance = 0.0f;
        float average = 0.0f;
	float maxLum = g_fMaxLum;
        
        for( int i = 0; i < 9; i++ )
        {
		luminance = g_tInputTex0.Sample( PointSampler, IN.vTexUV + g_vDSOffsets[i]).rg;
		average += luminance.r;
		maxLum = max(maxLum, luminance.g);
        }

	average /= 9.0f;
		
	// Return average luminance
        return float4( average, maxLum, 0.0f, 1.0f );
}

float4 PS_AdaptedLuminance( PS_INPUT IN ) : SV_Target
{
	// Get the current and last adapted luminance
	float2 fCurrentLum = g_tInputTex0.Sample( PointSampler, float2(0.5f, 0.5f)).rg;  
	float2 fLastLum = g_tInputTex1.Sample( PointSampler, float2(0.5f, 0.5f)).rg; 

	fCurrentLum = exp(fCurrentLum);	

	// Adapt the luminance using Pattanaik's technique  
	float2 fAdaptedLum = fLastLum + (fCurrentLum - fLastLum) * (1 - exp(-g_fDT * g_fTau));  

	return float4(fAdaptedLum.r, fAdaptedLum.g, 0.0f, 1.0f);   

}


//--------------------------------------------
// Techniques
//--------------------------------------------

//-----------------------------------------------------------------------------                             
// Desc: Performs initial downsample of the HDR texture
//-----------------------------------------------------------------------------
technique10 InitialDownSample
{
    pass P0
    {        
	SetVertexShader( CompileShader( vs_4_0, VS_QUAD() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_InitialDownSample() ) );
    }

}

//-----------------------------------------------------------------------------                             
// Desc: Downscales 3x3
//-----------------------------------------------------------------------------
technique10 DownSample
{
    pass P0
    {        
	SetVertexShader( CompileShader( vs_4_0, VS_QUAD() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_DownSample() ) );
    }

}

//-----------------------------------------------------------------------------                             
// Desc: Calculates adapted luminance
//-----------------------------------------------------------------------------
technique10 AdaptLuminance
{
    pass P0
    {        
	SetVertexShader( CompileShader( vs_4_0, VS_QUAD() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_AdaptedLuminance() ) );
    }

}