//--------------------------------------------------------------------------------------
// File: HDR.fx
//
// Chris Ravenscroft 2011
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Global variables
//--------------------------------------------------------------------------------------


texture2D g_tInputTex0;		// HDR Texture
texture2D g_tInputTex1;		// Adapted Luminance texture
texture2D g_tInputTex2;		// Bloom texture
texture2D g_tInputTex3;		// DirtyLens texture

cbuffer cbPerFrame
{
	float g_fExposure;
	bool g_bUseBloom;
	bool g_bUseDirtyLens;
};



//--------------------------------------------------------------------------------------
// Texture sampler
//--------------------------------------------------------------------------------------
SamplerState LinearSampler
{
    Filter = MIN_MAG_MIP_LINEAR;
    AddressU = Clamp;
    AddressV = Clamp;
};

SamplerState PointSampler
{
    Filter = MIN_MAG_MIP_POINT;
    AddressU = Clamp;
    AddressV = Clamp;
};

//--------------------------------------------------------------------------------------
// Vertex shader input structure
//--------------------------------------------------------------------------------------
struct VS_INPUT_QUAD
{
	float3 vPosition    : POSITION;
	float2 vTexUV       : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Pixel shader input structure
//--------------------------------------------------------------------------------------
struct PS_INPUT
{
	float4 vPositionCS : SV_POSITION;
	float2 vTexUV      : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Vertex Shaders
//--------------------------------------------------------------------------------------
PS_INPUT VS_QUAD( VS_INPUT_QUAD IN )
{
	PS_INPUT OUT;
	
	OUT.vPositionCS = float4(IN.vPosition, 1.0);
	OUT.vTexUV = IN.vTexUV;

	return OUT;
}


//--------------------------------------------
// Pixel Shaders
//--------------------------------------------

float4 PS_ToneMap( PS_INPUT IN ) : SV_Target
{
	// Sample the textures
	float4 vColour = g_tInputTex0.Sample(PointSampler, IN.vTexUV);
	float2 vLum = g_tInputTex1.Sample(PointSampler, float2(0.5,0.5)).rg;
	

	// Tone map / luminance using Reinhard's tone mapping equation
	// Lp is the luminance at the given pixel using the average scene luminance
	// Using Equation 2
	float Lp = (g_fExposure / vLum.r) * dot( vColour.rgb, float3(0.2125f, 0.7154f, 0.0721f) );

	// Using equation 3
	float toneScalar = ( Lp * ( 1.0f + ( Lp / (vLum.g * vLum.g) ) ) ) / ( 1.0f + Lp);

	// Tone map the final pixel
	vColour *= toneScalar;

	if(g_bUseBloom)
	{
		vColour += g_tInputTex2.Sample(LinearSampler, IN.vTexUV);
	}
	
	if(g_bUseDirtyLens)
	{
		vColour += g_tInputTex3.Sample(PointSampler, IN.vTexUV);;
	}
	
	vColour.a = 1.0f;

    return vColour;
}


//--------------------------------------------
// Techniques
//--------------------------------------------

//-----------------------------------------------------------------------------                             
// Desc: Tone maps the HDR values to LDR for display
//-----------------------------------------------------------------------------
technique10 ToneMap
{
    pass P0
    {        
	SetVertexShader( CompileShader( vs_4_0, VS_QUAD() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_ToneMap() ) );
    }

}