//--------------------------------------------------------------------------------------
// File: GBuffer.fx
//
// Chris Ravenscroft 2011
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Global variables
//--------------------------------------------------------------------------------------


Texture2D g_tDiffuse;
Texture2D g_tNormalMap;
Texture2D g_tSpecMap;

cbuffer cbPerFrame
{
	matrix g_mView;
	matrix g_mProjection;
	float g_fFarClip;
};

cbuffer cbPerModel
{
	float g_fSpecularPower;
	int g_iDiffuseMaterial;
};

cbuffer cbPerObject
{
	matrix g_mWorld;
};

//--------------------------------------------------------------------------------------
// Texture sampler
//--------------------------------------------------------------------------------------
SamplerState samLinear
{
    Filter = MIN_MAG_MIP_LINEAR;
    /*Filter = ANISOTROPIC;
    MaxAnisotropy = 8;*/
    AddressU = Wrap;
    AddressV = Wrap;
};

//--------------------------------------------------------------------------------------
// Vertex shader input structure
//--------------------------------------------------------------------------------------
struct VS_INPUT
{
	float3 vPosition    : POSITION;
	float3 vNormal	    : NORMAL;
	float4 vTangent	    : TANGENT;
	float3 vBiTangent   : BINORMAL;
	float2 vTexUV       : TEXCOORD0;
};

//--------------------------------------------------------------------------------------
// Pixel shader input structure
//--------------------------------------------------------------------------------------
struct PS_INPUT
{
	float4 vPositionCS : SV_POSITION;
	float3 vPositionVS : TEXCOORD0;
	float3 vNormalVS   : TEXCOORD1;
	float3 vTangentVS  : TEXCOORD3;
	float3 vBiTangentVS: TEXCOORD4;
	float2 vTexUV      : TEXCOORD2;
};

struct PS_OUT 
{
	float4 rt0   : SV_Target0;
	float4 rt1   : SV_Target1;
	float4 rt2   : SV_Target2;
};

PS_INPUT VS( VS_INPUT IN )
{
	PS_INPUT OUT;
	// View space position
	float4x4 mWorldView = mul(g_mWorld, g_mView);
    	float4 vPositionVS = mul(float4(IN.vPosition, 1.0f), mWorldView);
	OUT.vPositionVS = vPositionVS.xyz;

	// View space normal
	//float3x3 mWorldView3 = mul(g_mWorld, g_mView);
	OUT.vNormalVS = mul(float4(IN.vNormal,0.0f), mWorldView).xyz;
	OUT.vTangentVS = mul(IN.vTangent, mWorldView).xyz;
	OUT.vBiTangentVS = mul(float4(IN.vBiTangent,0.0f), mWorldView).xyz;

	// Tex
	OUT.vTexUV = IN.vTexUV;
	
	// Clip space position 
	OUT.vPositionCS = mul(vPositionVS, g_mProjection);
	

	return OUT;
}

PS_OUT PS( PS_INPUT IN )
{
	PS_OUT OUT;

	// Calculate the length of the view space position to get the distance from camera->surface
	OUT.rt0.r = IN.vPositionVS.z / g_fFarClip;
	OUT.rt0.g = g_fSpecularPower; // Specular power
	OUT.rt0.b = g_tSpecMap.Sample(samLinear, IN.vTexUV);// Specular intensity
	OUT.rt0.a = g_iDiffuseMaterial; // Diffuse material index


	//put normal into view space
	float3 bump = normalize(g_tNormalMap.Sample(samLinear, IN.vTexUV) * 2.0f - 1.0f);// normal map
	float3x3 TangentToView;
	TangentToView[0] = normalize(IN.vTangentVS);
	TangentToView[1] = normalize(IN.vBiTangentVS);
	TangentToView[2] = normalize(IN.vNormalVS);
	float3 normal = normalize(mul(bump, TangentToView)) * 0.5f + 0.5f;
	
	OUT.rt1.rgb = normal;
	OUT.rt1.a = 1.0f;

	OUT.rt2 = g_tDiffuse.Sample(samLinear, IN.vTexUV);

    	return OUT;
}

//--------------------------------------------------------------------------------------
// Techniques
//--------------------------------------------------------------------------------------
technique10 RenderGBuffer
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_4_0, VS() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS() ) );
    }
}