//--------------------------------------------------------------------------------------
// File: Bloom.fx
//
// Chris Ravenscroft 2011
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Global variables
//--------------------------------------------------------------------------------------


texture2D g_tInputTex0;		// Bloom texture
texture2D g_tInputTex1;		// Dirty Lens texture

cbuffer cbPerFrame
{
	float g_fMaxOpacity;				// Maximum alpha of lens texture
	float g_fBloomColourContrib;	// Amount of bloom colour that contributes to the "dirty lens" colour
};

//--------------------------------------------------------------------------------------
// Texture sampler
//--------------------------------------------------------------------------------------
SamplerState LinearSampler
{
    Filter = MIN_MAG_MIP_LINEAR;
    AddressU = Clamp;
    AddressV = Clamp;
};

SamplerState PointSampler
{
    Filter = MIN_MAG_MIP_POINT;
    AddressU = Clamp;
    AddressV = Clamp;
};

//--------------------------------------------------------------------------------------
// Vertex shader input structure
//--------------------------------------------------------------------------------------
struct VS_INPUT_QUAD
{
	float3 vPosition    : POSITION;
	float2 vTexUV       : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Pixel shader input structure
//--------------------------------------------------------------------------------------
struct PS_INPUT
{
	float4 vPositionCS : SV_POSITION;
	float2 vTexUV      : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Vertex Shaders
//--------------------------------------------------------------------------------------
PS_INPUT VS_QUAD( VS_INPUT_QUAD IN )
{
	PS_INPUT OUT;
	
	OUT.vPositionCS = float4(IN.vPosition, 1.0);
	OUT.vTexUV = IN.vTexUV;

	return OUT;
}


//--------------------------------------------
// Pixel Shaders
//--------------------------------------------

float4 PS_DirtyLens( PS_INPUT IN ) : SV_Target
{
	// Sample lens texture and clamp the max opacity
	float4 vLensTex = g_tInputTex1.Sample(PointSampler, IN.vTexUV);
	vLensTex = min(vLensTex, float4(g_fMaxOpacity, g_fMaxOpacity, g_fMaxOpacity, 1.0f));
	
	// Flip the bloom texture and multiply with the lens texture
	float2 vFlippedUV = -IN.vTexUV + 1.0f;
	float4 vLensDirt = g_tInputTex0.Sample(LinearSampler, vFlippedUV);
	vLensDirt = min(vLensDirt, float4(1.0f, 1.0f, 1.0f, 1.0f)) * g_fBloomColourContrib;
	
	float4 vColour = vLensTex * vLensDirt;
	vColour.w = 1.0f;
	return vColour;
}

//--------------------------------------------
// Techniques
//--------------------------------------------

//-----------------------------------------------------------------------------                             
// Desc: Creates the dirty lens effect
//-----------------------------------------------------------------------------
technique10 DirtyLens
{
    pass P0
    {        
		SetVertexShader( CompileShader( vs_4_0, VS_QUAD() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_DirtyLens() ) );
    }

}