//--------------------------------------------------------------------------------------
// File: DepthOfField.fx
//
// Chris Ravenscroft 2011
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Global variables
//--------------------------------------------------------------------------------------


static const int NUM_DOF_TAPS = 12;
static const float MAX_COC = 5.0f;

texture2D g_tInputTex0;		// Depth texture
texture2D g_tInputTex1;		// HDR texture


cbuffer cbComputeDepthBlur
{
	float g_fFarFocus;
	float g_fNearFocus;
	float g_fFarBlur;
	float g_fNearBlur;
	float g_fFarBlurClamp;
	float g_fNearBlurClamp;
	float2 g_vFilterTaps[NUM_DOF_TAPS];
	float g_fFarClip;
};


//--------------------------------------------------------------------------------------
// Texture sampler
//--------------------------------------------------------------------------------------
SamplerState LinearSampler
{
    Filter = MIN_MAG_MIP_LINEAR;
    AddressU = Clamp;
    AddressV = Clamp;
};

SamplerState PointSampler
{
    Filter = MIN_MAG_MIP_POINT;
    AddressU = Clamp;
    AddressV = Clamp;
};

//--------------------------------------------------------------------------------------
// Vertex shader input structure
//--------------------------------------------------------------------------------------
struct VS_INPUT_QUAD
{
	float3 vPosition    : POSITION;
	float2 vTexUV       : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Pixel shader input structure
//--------------------------------------------------------------------------------------
struct PS_INPUT
{
	float4 vPositionCS : SV_POSITION;
	float2 vTexUV      : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Vertex Shaders
//--------------------------------------------------------------------------------------
PS_INPUT VS_QUAD( VS_INPUT_QUAD IN )
{
	PS_INPUT OUT;
	
	OUT.vPositionCS = float4(IN.vPosition, 1.0);
	OUT.vTexUV = IN.vTexUV;

	return OUT;
}


//--------------------------------------------
// Pixel Shaders
//--------------------------------------------


float ComputeDepthBlur( float fDepth )
{
	float b = 0.0f;

	if( fDepth < g_fNearFocus)
	{
		b = (g_fNearFocus - fDepth) / (g_fNearFocus - g_fNearBlur);
		b = clamp(b, 0.0f, g_fNearBlurClamp);
	}
	else
	{
		b = (fDepth - g_fFarFocus) / (g_fFarBlur - g_fFarFocus);
		b = clamp(b, 0.0f, g_fFarBlurClamp);
	}

	return b;
}


float4 PS_PDiscDOF( PS_INPUT IN ) : SV_Target
{
	// Start with center sample color
	float4 vColour = g_tInputTex1.Sample(PointSampler, IN.vTexUV);
	float fTotalContribution = 1.0f;

	// Depth and blurriness values for center sample
	float fCenterDepth = g_tInputTex0.Sample(PointSampler, IN.vTexUV).r * g_fFarClip;
	float fCenterBlur = ComputeDepthBlur(fCenterDepth);

	if (fCenterBlur > 0)
	{
		// Compute CoC size based on blurriness
		float fSizeCoC = fCenterBlur * MAX_COC;

		// Run through all filter taps
		[unroll(NUM_DOF_TAPS)]
		for (int i = 0; i < NUM_DOF_TAPS; i++)
		{
			// Compute sample coordinates
			float2 vTapCoord = IN.vTexUV + g_vFilterTaps[i] * fSizeCoC;

			// Fetch filter tap sample
			float4 vTapColour = g_tInputTex1.SampleLevel(PointSampler, vTapCoord, 0);
			float fTapDepth = g_tInputTex0.SampleLevel(PointSampler, vTapCoord, 0).r * g_fFarClip;
			float fTapBlur = ComputeDepthBlur(fTapDepth);

			// Compute tap contribution based on depth and blurriness
			float fTapContribution = (fTapDepth > fCenterDepth) ? 1.0f : fTapBlur;

			// Accumulate color and sample contribution
			vColour += vTapColour * fTapContribution;
			fTotalContribution += fTapContribution;
		}
	}

	// Normalize colour
	float4 vFinalColour = vColour / fTotalContribution;
	vFinalColour.w = 1.0f;
	return vFinalColour;
}


//--------------------------------------------
// Techniques
//--------------------------------------------


//-----------------------------------------------------------------------------                             
// Desc: Create depth of field effect with a poison disc blur
//-----------------------------------------------------------------------------
technique10 PDiscDOF
{
    pass P0
    {        
	SetVertexShader( CompileShader( vs_4_0, VS_QUAD() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_PDiscDOF() ) );
    }

}
