//--------------------------------------------------------------------------------------
// File: Bloom.fx
//
// Chris Ravenscroft 2011
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Global variables
//--------------------------------------------------------------------------------------


texture2D g_tInputTex0;		// Source texture
texture2D g_tInputTex1;		// Luminance texture

float2 g_vDSOffsets[4];		// Offsets to downscale a texture


cbuffer cbBrightPass
{
	float g_fThreshold;
	float g_fExposure;
};

cbuffer cbBlur
{
	float2 g_vBlurOffsets[9];	// Offsets to blur a texture
	float g_fBlurWeights[9];	// Blur weights
	float g_fWeight;		// Bloom weighting
};

//--------------------------------------------------------------------------------------
// Texture sampler
//--------------------------------------------------------------------------------------
SamplerState LinearSampler
{
    Filter = MIN_MAG_MIP_LINEAR;
    AddressU = Clamp;
    AddressV = Clamp;
};

SamplerState PointSampler
{
    Filter = MIN_MAG_MIP_POINT;
    AddressU = Clamp;
    AddressV = Clamp;
};

//--------------------------------------------------------------------------------------
// Vertex shader input structure
//--------------------------------------------------------------------------------------
struct VS_INPUT_QUAD
{
	float3 vPosition    : POSITION;
	float2 vTexUV       : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Pixel shader input structure
//--------------------------------------------------------------------------------------
struct PS_INPUT
{
	float4 vPositionCS : SV_POSITION;
	float2 vTexUV      : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Vertex Shaders
//--------------------------------------------------------------------------------------
PS_INPUT VS_QUAD( VS_INPUT_QUAD IN )
{
	PS_INPUT OUT;
	
	OUT.vPositionCS = float4(IN.vPosition, 1.0);
	OUT.vTexUV = IN.vTexUV;

	return OUT;
}


//--------------------------------------------
// Pixel Shaders
//--------------------------------------------

float4 PS_Downscale4x( PS_INPUT IN ) : SV_Target
{
	float4 vSample = 0.0f;

	for(int i = 0; i < 4; i++)
	{
		vSample += g_tInputTex0.Sample(PointSampler, IN.vTexUV + g_vDSOffsets[i]);
	}

	vSample *= 0.25f;

	return vSample;
}

float4 PS_ApplyThreshold( PS_INPUT IN ) : SV_Target
{
	// Sample the textures
	float4 vColour = g_tInputTex0.Sample(PointSampler, IN.vTexUV);
	float2 vLum = g_tInputTex1.Sample(PointSampler, float2(0.5,0.5)).rg;

	// Apply bright threshold
	vColour.rgb -= g_fThreshold;
	vColour = max(vColour, 0.0f);

	// tone map / luminance using Reinhard's tone mapping equation
	// Lp is the luminance at the given pixel using the average scene luminance
	float Lp = (g_fExposure / vLum.r) * dot( vColour.rgb, float3(0.2125f, 0.7154f, 0.0721f) );

	// Using equation 3
	float toneScalar = ( Lp * ( 1.0f + ( Lp / (vLum.g * vLum.g) ) ) ) / ( 1.0f + Lp);

	// Tone map the pixel
	vColour *= toneScalar;
	vColour.rgb /= (1.0f + vColour);

	return vColour;
}


float4 PS_Blur( PS_INPUT IN ) : SV_Target
{
	float4 vColour = 0.0f;

	for( int i = 0; i < 9; i++ )
	{
		vColour += g_tInputTex0.Sample(PointSampler, IN.vTexUV + g_vBlurOffsets[i]) * g_fBlurWeights[i];
	}
	
	vColour *= g_fWeight;
	return vColour;
}


float4 PS_Upscale4x( PS_INPUT IN ) : SV_Target
{
	float4 vSample = g_tInputTex0.Sample(LinearSampler, IN.vTexUV);

	return vSample;
}


//--------------------------------------------
// Techniques
//--------------------------------------------

//-----------------------------------------------------------------------------                             
// Desc: Downscales source texture 2x
//-----------------------------------------------------------------------------
technique10 Downscale4x
{
    pass P0
    {        
	SetVertexShader( CompileShader( vs_4_0, VS_QUAD() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_Downscale4x() ) );
    }

}

//-----------------------------------------------------------------------------                             
// Desc: Tone map and apply bright threshold
//-----------------------------------------------------------------------------
technique10 BrightPass
{
    pass P0
    {        
	SetVertexShader( CompileShader( vs_4_0, VS_QUAD() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_ApplyThreshold() ) );
    }

}

//-----------------------------------------------------------------------------                             
// Desc: Blur source texture
//-----------------------------------------------------------------------------
technique10 Blur
{
    pass P0
    {        
	SetVertexShader( CompileShader( vs_4_0, VS_QUAD() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_Blur() ) );
    }

}


//-----------------------------------------------------------------------------                             
// Desc: Upscales source texture 2x
//-----------------------------------------------------------------------------
technique10 Upscale4x
{
    pass P0
    {        
	SetVertexShader( CompileShader( vs_4_0, VS_QUAD() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_Upscale4x() ) );
    }

}